# https://www.instana.com/blog/stans-robot-shop-sample-microservice-application/
# https://github.com/instana/robot-shop

from pyparsing import *
from detectors.pyparsingPatterns import *
from detectors.project import *
from detectors.detectorContext import *
from detectors.evidences import *

project = Project("Microservice Based Robot Shop", "../systems/robot-shop-master")
# TODO: maybe evidences + is still in the docker compose file, is still deployed in K8S
# TODO: other functionality possible: record all files belonging to component / traceability; -> then list all files not classified.
# TODO: other functionality possible: get package deps from package.json/docker => which services is this using?
# TODO: other functionality possible: NPM module stereotype
# TODO:  call relations in .js
# TODO: relation to mongo db
cart = project.createComponent("Cart Service", "service", "./cart", [FilesPresent(["package.json", "Dockerfile"]), AtLeastOnePresent("*.js")])
catalogue = project.createComponent("Catalogue Service", "service", "./catalogue", [FilesPresent(["package.json", "Dockerfile"]), AtLeastOnePresent("*.js")])

# TODO: DC/OS dir => deployment view

# uses rabit mq ... in docker compose it composes ../payment into it ... uses instana
dispatch = project.createComponent("Dispatch Service", "service", "dispatch", AtLeastOnePresent("./*/*.go"))

# TODO: K8S dir => deployment view

# TODO: load gen -> client?

# TODO: mongo / mysql

# TODO: openshif dir => deployment view

# has also its own docker compose file, uses rabit mq
payment = project.createComponent("Payment Service", "service", "payment", AtLeastOnePresent("*.py"))

ratings = project.createComponent("Ratings Service", "service", "ratings", AtLeastOnePresent("html*/*.php"))

shipping = project.createComponent("Shipping Service", "service", "shipping", [FilesPresent("pom.xml"), AtLeastOnePresent("src/*/*.java")])

# TODO: Swarm => deployment view

user = project.createComponent("User Service", "service", "./user", [FilesPresent(["package.json", "Dockerfile"]), AtLeastOnePresent("*.js")])

apiGateway = project.createComponent("NGINX API Gateway", ["service", "facade"], "./web", 
    [FilesPresent(["default.conf.template"]), 
    FileContains("Dockerfile", ["(FROM\\s*nginx)", "EXPOSE\\s*"])])

failedEvidencesCount = len(project.failedEvidences)

def nginxLocationDetector(detectorContext, **kwargs):
    print("************ nginx location DETECTORS ************** ")
    args = getArgsFromKwArgs(['location', 'host'], **kwargs)

    detectorContext.containsCurlyBracesBlock(Literal("location") + Literal(args["location"])).matchesPattern(Literal("proxy_pass") + Regex("http://.*" + args["host"] + "[^;]*") + Literal(";"))

def dockerfileEnvVarDectector(detectorContext, **kwargs):
    print("************ dockerfileEnvVarDectector DETECTORS ************** ")
    args = getArgsFromKwArgs(['envVar'], **kwargs)

    detectorContext.matchesPattern(Literal("ENV") + OneOrMore(Word(printables, excludeChars = "=\\") + 
        Literal("=") + Word(printables, excludeChars = "=\\") + Optional("\\"))).matchesPattern(Literal(args["envVar"]) + 
        Literal("=") + Word(printables, excludeChars = "=\\"))
       
for name, componentID, hostEnvVar in [
    ["catalogue", catalogue, "CATALOGUE_HOST"],
    ["user", user, "USER_HOST"],
    ["cart", cart, "CART_HOST"],
    ["shipping", shipping, "SHIPPING_HOST"],
    ["payment", payment, "PAYMENT_HOST"],
    ["ratings", ratings, "RATINGS_HOST"]]:
    project.createLinks(apiGateway, componentID, {"roleName": "target", "stereotypeInstances": "restfulHTTP"}, 
        "./web", [DetectInFile("default.conf.template", nginxLocationDetector, 
            location=f"/api/{name}/", host=hostEnvVar),
            DetectInFile("Dockerfile", dockerfileEnvVarDectector, 
            envVar=hostEnvVar)])


webClient = project.createComponent("REST API Client", "client", None, NoNewFailedEvidences(failedEvidencesCount, "NGINX locations for API elements"))
project.createLinks(webClient, apiGateway, {"roleName": "target", "stereotypeInstances": "restfulHTTP"}, None, NoNewFailedEvidences(failedEvidencesCount, "NGINX locations for API elements")) 

failedEvidencesCount = len(project.failedEvidences)
webUI = project.createComponent("Web Client", "webUI", "./web", [FilesPresent(["default.conf.template"]), AtLeastOnePresent(["*static/js/*.js", "*static/*.html"])])
project.createLinks(webUI, apiGateway, {"roleName": "target", "stereotypeInstances": "http"}, None, NoNewFailedEvidences(failedEvidencesCount, "Web interface exposed through NGINX proxy")) 

#project.createLinks(cart, catalogue, {"roleName": "target", "stereotypeInstances": "restfulHTTP"}, "./cart", [JSAppCallsRequestMethod("server.js", "catalogue")]) 

def cartToCatalogueLinkDetector(detectorContext):
    print("************ app.get DETECTORS ************** ")
    detectorContext.containsObjectCallTo("app", "get").containsProcCallTo("getProduct")
    print("************ getP DETECTORS ************** ")
    # the matches pattern for the promise indicates the callback based invocation, the request the service invocation in it
    detectorContext.containsJsFunction("getProduct").matchesPattern(Literal("return") + Literal("new") + 
        Literal("Promise") + roundBracesBlock + Literal(";")).containsProcCallTo("request").matchesRegex("^['\"]http://.*catalogueHost")

project.createLinks(cart, catalogue, {"roleName": "target", "stereotypeInstances": "restfulHTTP, callback"}, "./cart", 
    DetectInFile("server.js", cartToCatalogueLinkDetector)) 

def redisDBFromJSDetector(detectorContext, **kwargs):
    print("************ redisDBFromJSDetector ************** ")
    args = getArgsFromKwArgs(['redisVar', 'redisClientVar'], **kwargs)
    detectorContext.matchesPattern(Literal(args["redisVar"]) + Literal("=") + Literal("require") + Literal("(") + Literal("'") + Literal("redis") + "'" + Literal(")"))
    detectorContext.matchesPattern(Literal(args["redisClientVar"]) + Literal("=") + Literal(args["redisVar"]) + Literal(".") + Literal("createClient"))
    detectorContext.matchesPattern(Literal(args["redisClientVar"]) + Literal(".") +  oneOf(["setex", "get", "del"]))
    # detect that at least one async invocation in a Promise is present too
    detectorContext.matchesPattern(Literal("return") + Literal("new") + 
        Literal("Promise") + roundBracesBlock + Literal(";")).matchesPattern(Literal(args["redisClientVar"]) + 
        Literal(".") +  oneOf("setex get del"))

def dockerComposeImageBasedService(ctx, **kwargs):
    print("************ goFuncToConsumerRabbitChannel ************** ")
    args = getArgsFromKwArgs(["serviceName", "imageNamePrefix"], **kwargs)
    ctx.matchIndentedBlock(Literal("services") + Literal(":"), "no services block in docker compose file").matchIndentedBlock(
        Literal(args["serviceName"]) + Literal(":"), "service '" + args["serviceName"] + 
        "' block missing in services section of docker compose file").matchesPattern(Literal("image") + Literal(":") +
        Literal(args["imageNamePrefix"]))

cartAndAnonymousUserCountDB = project.createComponent("Cart and Anonymous User Count DB", "redisDB", "./",
    DetectInFile("docker-compose.yaml", dockerComposeImageBasedService, serviceName = "redis", imageNamePrefix = "redis"))

project.createLinks(cart, cartAndAnonymousUserCountDB, {"roleName": "target", "stereotypeInstances": "resp, synchronousConnector, callback", 
    "taggedValues": "'description': 'async callback for saving cart, sync get, del, etc. for all other db operations'"}, 
    "./cart", 
    DetectInFile("server.js", redisDBFromJSDetector, redisVar = "redis", redisClientVar = "redisClient"))

def jsPrometheusImportDetector(ctx):
    ctx.containsProcCallTo("require").matchesPattern(Word("'`\"") + Literal("prom-client") + Word("'`\""))
failedEvidencesCount = len(project.failedEvidences)
cartPrometheus = project.createComponent("Cart Prometheus Monitor", "monitoringComponent", "./cart", 
    DetectInFile("server.js", jsPrometheusImportDetector))
project.createLinks(cart, cartPrometheus, {"roleName": "target", "stereotypeInstances": "inMemoryConnector"}, 
    None, NoNewFailedEvidences(failedEvidencesCount, "cart prometheus link"))

def dockerComposeContainsServiceBlock(ctx, **kwargs):
    args = getArgsFromKwArgs(["serviceName"], **kwargs)
    ctx.matchIndentedBlock(Literal("services") + Literal(":"), "no services block in docker compose file").matchIndentedBlock(
            Literal(args["serviceName"]) + Literal(":"), "no mongo db service in docker compose file")

def dockerfileContainsFromStatement(ctx, **kwargs):
    args = getArgsFromKwArgs(["image"], **kwargs)
    ctx.matchesPattern(Literal("FROM") + Literal(args["image"]) + Literal(":"))

catalogueUsersOrdersDb = project.createComponent("Catalogue Users Orders DB", "mongoDB", "./", 
    [DetectInFile("docker-compose.yaml", dockerComposeContainsServiceBlock, serviceName = "mongodb"), 
    DetectInFile("./mongo/Dockerfile", dockerfileContainsFromStatement, image = "mongo")])

def mongoDBFromJSDetector(detectorContext, **kwargs):
    print("************ mongoDBFromJSDetector ************** ")
    args = getArgsFromKwArgs(['mongoClientVar', 'mongoCollectionVar', 'collectionName'], **kwargs)
    detectorContext.matchesPattern(Literal(args["mongoClientVar"]) + Literal("=") + Literal("require") + Literal("(") + Literal("'") + Literal("mongodb") + "'" + Literal(")") + Literal(".") + Literal("MongoClient"))
    detectorContext.matchesPattern(Literal(args["mongoClientVar"]) + Literal(".") + Literal("connect") + roundBracesBlock + 
        Literal(";")).matchesPattern(Literal(args["mongoCollectionVar"]) + Literal("=") + ID + Literal(".") + Literal("collection") + 
        roundBracesBlock).matchesPattern(Word("'`\"") + Literal(args["collectionName"]) + Word("'`\""))
    detectorContext.matchesPattern(Literal(args["mongoCollectionVar"]) + Literal(".") + oneOf(["find", "findOne", "distinct"]))
    
project.createLinks(catalogue, catalogueUsersOrdersDb, {"roleName": "target", "stereotypeInstances": "mongoWire, callback"}, 
    "./catalogue", DetectInFile("server.js", mongoDBFromJSDetector, mongoClientVar = "mongoClient", mongoCollectionVar = "collection",
    collectionName = "products"))

print("************ detect payment gateway **************")
failedEvidencesCount = len(project.failedEvidences)
# evidence is a get as this is only a dummy call to paypal in the source code for now
paymentGateway = project.createComponent("Paypal Payment Gateway", ["service", "externalComponent"], "./payment", DetectInFile("payment.py",  
    lambda ctx: ctx.containsObjectCallTo("requests", "get").matchesPattern(Literal("PAYMENT_GATEWAY"))))
project.createLinks(payment, paymentGateway, {"roleName": "target", "stereotypeInstances": "restfulHTTP, synchronousConnector"}, 
    None, NoNewFailedEvidences(failedEvidencesCount, "link payment to payment gateway"))

def catalogueLinksDetector(detectorContext, **kwargs):
    print("************ catalogueLinksDetector ************** ")
    args = getArgsFromKwArgs(["target"], **kwargs)
    # TODO: we could also check that this is in a method that has an app.route annotation, but lets keep it simple for now
    detectorContext.matchesPattern(pythonRequestsCallPattern).matchesPatterns([Literal('http://'), Literal(args["target"])])

project.createLinks(payment, user, {"roleName": "target", "stereotypeInstances": "restfulHTTP, synchronousConnector", "label": "check for users and post orders"}, "./payment", 
    DetectInFile("payment.py", catalogueLinksDetector, target = "USER")) 

project.createLinks(payment, cart, {"roleName": "target", "stereotypeInstances": "restfulHTTP, synchronousConnector"}, "./payment", 
    DetectInFile("payment.py", catalogueLinksDetector, target = "CART")) 

def pythonPrometheusImportDetector(ctx):
    ctx.matchesPattern(Literal("import") + Literal("prometheus_client"))
failedEvidencesCount = len(project.failedEvidences)
paymentPrometheus = project.createComponent("Payment Prometheus Monitor", "monitoringComponent", "./payment", 
    DetectInFile("payment.py", pythonPrometheusImportDetector))
project.createLinks(payment, paymentPrometheus, {"roleName": "target", "stereotypeInstances": "inMemoryConnector"}, 
    None, NoNewFailedEvidences(failedEvidencesCount, "payment prometheus link"))



#
# Instana / Open Tracing Based Tracing
#

# there is no real evidence for the instana component (started outside of the project) other than the links
# to it provided below (which would fail if instana is not used anymore)
instana = project.createComponent("Instana Agent", "tracingComponent", None, None)

def jsRequireInstanaDetector(ctx):
    print("************ jsRequireInstanaDetector ************** ")
    ctx.containsProcCallTo("require").matchesPattern(Literal("@instana/collector"))

project.createLinks(cart, instana, {"roleName": "target", "stereotypeInstances": "http2"}, "./cart", 
    DetectInFile("server.js", jsRequireInstanaDetector)) 

project.createLinks(catalogue, instana, {"roleName": "target", "stereotypeInstances": "http2"}, "./catalogue", 
    DetectInFile("server.js", jsRequireInstanaDetector)) 

project.createLinks(dispatch, instana, {"roleName": "target", "stereotypeInstances": "http2"}, "./dispatch/src", 
    DetectInFile("main.go", lambda ctx: ctx.containsProcCallTo("import").
    matchesPatterns([Literal("github.com/instana/go-sensor"), Literal("github.com/opentracing/opentracing-go")])))

project.createLinks(payment, instana, {"roleName": "target", "stereotypeInstances": "http2"}, "./payment", 
    DetectInFile("payment.py", lambda ctx: ctx.matchesPattern(Literal("import") + Literal ("opentracing")))) 

project.createLinks(user, instana, {"roleName": "target", "stereotypeInstances": "http2"}, "./user", 
    DetectInFile("server.js", jsRequireInstanaDetector)) 




def pythonFunctionContainingCallDetector(ctx, **kwargs):
    print("************ pythonFunctionContainingCallDetector ************** ")
    args = getArgsFromKwArgs(["functionName", "callObject", "callMethod"], **kwargs)
    errorMessage = "python function '" + args["functionName"] + "' containing call to '" + args["callObject"] + "." + args["callMethod"] + "' not detected"
    ctx.matchIndentedPythonBlock(Literal("def") + Literal(args["functionName"]) + roundBracesBlock + Literal(":"), 
        errorMessage).containsObjectCallTo(args["callObject"], args["callMethod"])

def goFuncToConsumerRabbitChannel(ctx, **kwargs):
    print("************ goFuncToConsumerRabbitChannel ************** ")
    args = getArgsFromKwArgs(["rabbitChanVar", "queueName"], **kwargs)
    ctx.matchesPattern(Literal("go") + Literal("func") + roundBracesBlock + 
        curlyBracesBlock).containsObjectCallTo(args["rabbitChanVar"], "Consume").matchesPattern(
        Literal('"') + Literal(args["queueName"]) + Literal('"'))

# project.createLinks(payment, dispatch, {"roleName": "target", "stereotypeInstances": "pubSubConnector, messaging", 
#     "label": "order processing queue", "taggedValues": "'publishers': [paymentService], 'subscribers': [dispatchService]"}, "./", 
#     [DetectInFile("./payment/payment.py", pythonFunctionContainingCallDetector, functionName = "queueOrder", 
#     callObject = "publisher", callMethod = "publish"), DetectInFile("./dispatch/src/main.go", 
#     goFuncToConsumerRabbitChannel, rabbitChanVar = "rabbitChan", 
#     queueName = "orders")])

rabbitMQ = project.createComponent("Rabbit MQ", ["messageBroker", "pubSubComponent"], "./dispatch", 
    DetectInFile("docker-compose.yaml", dockerComposeImageBasedService, serviceName = "rabbitmq", 
    imageNamePrefix = "rabbitmq"))

project.createLinks(payment, rabbitMQ, {"roleName": "target", "stereotypeInstances": "publisher", 
    "taggedValues": "'channel': 'orders'"}, "./payment", 
    [DetectInFile("payment.py", pythonFunctionContainingCallDetector, functionName = "queueOrder", 
    callObject = "publisher", callMethod = "publish")])
    
project.createLinks(dispatch, rabbitMQ, {"roleName": "target", "stereotypeInstances": "subscriber", 
    "taggedValues": "'channel': 'orders'"}, "./dispatch", 
    [DetectInFile("./src/main.go", 
    goFuncToConsumerRabbitChannel, rabbitChanVar = "rabbitChan", 
    queueName = "orders")])


def dockerComposeImageBasedService(ctx, **kwargs):
    print("************ goFuncToConsumerRabbitChannel ************** ")
    args = getArgsFromKwArgs(["serviceName", "imageNamePrefix"], **kwargs)
    ctx.matchIndentedBlock(Literal("services") + Literal(":"), "no services block in docker compose file").matchIndentedBlock(
        Literal(args["serviceName"]) + Literal(":"), "service '" + args["serviceName"] + 
        "' block missing in services section of docker compose file").matchesPattern(Literal("image") + Literal(":") +
        Literal(args["imageNamePrefix"]))
    
ratingsAndCitiesDB = project.createComponent("Ratings and Shipping Cities DB", "mySQLDB", "./", 
    [DetectInFile("docker-compose.yaml", dockerComposeContainsServiceBlock, serviceName = "mysql"),
    DetectInFile("mysql/Dockerfile", dockerfileContainsFromStatement, image = "mysql")])

def connectPhpRatingsToMySQLRatingsDB(ctx):
    print("************ connectPhpToMySQLRatingsDB ************** ")
    ctx.matchesPattern(Literal("function") + Literal("_dbConnect") + roundBracesBlock + curlyBracesBlock).matchesPatterns(
        [Literal("mysql") + Literal(":"), Literal("dbname") + Literal("=") + Literal("ratings")])

project.createLinks(ratings, ratingsAndCitiesDB, {"roleName": "target", "stereotypeInstances": "mySQLProtocol"}, 
    "./ratings/html", DetectInFile("api.php", connectPhpRatingsToMySQLRatingsDB))

def connectPhpRatingsToCatalogueService(ctx):
    print("************ connectPhpRatingsToCatalogueService ************** ")
    ctx.matchesPattern(Literal("function") + Literal("_checkSku") + roundBracesBlock + curlyBracesBlock).matchesPatterns(
        [Literal("http://catalogue")])

project.createLinks(ratings, catalogue, {"roleName": "target", "stereotypeInstances": "restfulHTTP"}, 
    "./ratings/html", DetectInFile("api.php", connectPhpRatingsToCatalogueService))

project.createLinks(user, catalogueUsersOrdersDb, {"roleName": "target", "stereotypeInstances": "mongoWire, callback", "label": "users and orders"}, 
    "./user", [DetectInFile("server.js", mongoDBFromJSDetector, mongoClientVar = "mongoClient", mongoCollectionVar = "usersCollection",
    collectionName = "users"), DetectInFile("server.js", mongoDBFromJSDetector, mongoClientVar = "mongoClient", mongoCollectionVar = "ordersCollection",
    collectionName = "orders")])

project.createLinks(user, cartAndAnonymousUserCountDB, {"roleName": "target", "stereotypeInstances": "resp, callback", 
    "label": "only used to track anonymous users"}, 
    "./cart", 
    DetectInFile("server.js", redisDBFromJSDetector, redisVar = "redis", redisClientVar = "redisClient"))

def javaCartHTTPPost(ctx):
    ctx.matchesPattern(Literal("CART_URL") + Literal("=") + OneOrMore(Word(printables, excludeChars = ";"))).matchesPatterns(
        [Literal("http://"), Literal("/shipping"), Literal("CART_ENDPOINT")])
    ctx.matchesPattern(Literal("new") + Literal("HttpPost") + roundBracesBlock).matchesPattern(Literal("CART_URL"))
    
project.createLinks(shipping, cart, {"roleName": "target", "stereotypeInstances": "restfulHTTP"}, 
    "./shipping/src/main/java/org/steveww/spark/", DetectInFile("Main.java", javaCartHTTPPost))

def javaMySQLConnection(ctx):
    ctx.matchesPattern(Literal("JDBC_URL ") + Literal("=") + OneOrMore(Word(printables, excludeChars = ";"))).matchesPatterns(
        [Literal("jdbc:"), Literal("mysql:"), Literal("DB_HOST")])
    
project.createLinks(shipping, ratingsAndCitiesDB, {"roleName": "target", "stereotypeInstances": "jdbc", "label" : "read only, query predefined cities table"}, 
    "./shipping/src/main/java/org/steveww/spark/", DetectInFile("Main.java", javaMySQLConnection))



if project.failedEvidences != []:
    print("*** FAILED EVIDENCES / MODEL ELEMENTS NOT ADDED ***\n")
    no = 0
    for evidence in project.failedEvidences:
        no = no + 1
        print(str(no) + ": " + evidence)
    print("\n*** \n")

# print("*** RESULTING MODEL ***\n")
# print(project.genModel)
# print("*** \n")

project.saveAsFile("genmodel.py")

import subprocess
subprocess.call(['python3', 'generateAll.py'])


